# -*- coding: iso-8859-1 -*-

import engine.analyze
import data.analysis
import data.text
import lang.Hyphenation
import config.ids
import config.lang
import main.TestDataReader
import unittest

class TestAnalyzer(unittest.TestCase):

    def setUp(self):
        self.hyph = lang.Hyphenation.Hyphenation(
                config.lang.vowels, 
                config.lang.consonants, 
                config.lang.diphNormal, 
                config.lang.diphSpecial, 
                config.lang.customLower, 
                config.lang.customUpper, 
                config.lang.inputPartSeparator, 
                config.lang.outputPartSeparator, 
                config.lang.outputSyllSeparator, 
                config.lang.syllShapes,
                config.ids.maxSyllabifications,
                config.ids.penalty)
        self.analyzer = engine.analyze.Analyzer(self.hyph)
        
    def testValidateSyllLen(self):
        self.doTest('syllable_length.txt')

    def testValidateSyllLenLong(self):
        self.doTest('syllable_length_long_line.txt')
    
    def testValidateMono(self):
        self.doYesNoTest('monosyllabic_final.txt')
    
    def testValidateLongVowel(self):
        self.doYesNoTest('long_vowel_last_syll.txt')
    
    def testCaesura(self):
        self.doYesNoTest('caesura.txt')
    
    def testHeavyBeginning(self):
        self.doGenericCommentTest('heavy_beginning.txt')        
                
    def testStoreAlliteration(self):
        self.doAllitTest('alliteration.txt')
    
    def testStatsMainType(self):
        result = self.doSimpleTest('joo e-li tes-ti-li puk-kaa')
        self.assertEqual('1232', result.mainType)
    
    def testStatsMainTypeCompound(self):
        result = self.doSimpleTest('ku-pa-ri-sen um-pi=pur-ren')
        self.assertEqual('44', result.mainType)
    
    def testStatsMainTypeShortLong(self):
        result = self.doSimpleTest('pah')
        self.assertEqual('1', result.mainType)
        result = self.doSimpleTest('a-a-a-a-a')
        self.assertEqual('5', result.mainType)
                
    def doAllitTest(self, testFile):
        reader = main.TestDataReader.TestDataReader(
                'engine/testdata/' + testFile)
        testData = reader.getPairs()

        for input, expected in testData:
            line = data.analysis.Line()
            line.content = data.text.Text(input)
            output = data.analysis.AnalysisResult()
            self.analyzer.storeAlliteration(line, output)
            self.assertEqual(expected, output.alliterationTypes)
             
    def doTest(self, testFile):
        reader = main.TestDataReader.TestDataReader(
                'engine/testdata/' + testFile)
        testData = reader.getPairs()

        for data, result in testData:
            self.assertEqual(result, self.getResult(data))

    def doYesNoTest(self, testFile):
        reader = main.TestDataReader.TestDataReader(
                'engine/testdata/' + testFile)
        testData = reader.getPairs()

        for data, result in testData:
            actualResult = self.getYesNoResult(data)
            self.assertEqual(result, actualResult, 
                             "%s: %s != %s" 
                             % (data, actualResult, result))
                                 
    def doGenericCommentTest(self, testFile):
        reader = main.TestDataReader.TestDataReader(
                'engine/testdata/' + testFile)
        testData = reader.getPairs()

        for input, expected in testData:
            line = data.analysis.Line()
            line.content = data.text.Text(input)
            result = self.analyzer.validate(line)
            comments = ';'.join(result.generalErrors.keys()
                                + result.warnings.keys())
            self.assertEqual(expected, comments, 
                             'input=%s,expected=%s,result=%s'
                             % (input, expected, comments))
            
    def doSimpleTest(self, text):
        line = data.analysis.Line()
        line.content = data.text.Text(text)
        return self.analyzer.validate(line)
        
    def getResult(self, theData):
        line = data.analysis.Line()
        line.content = data.text.Text(theData)
        result = self.analyzer.validate(line)
        shortErr = [str(i) for i in \
                result.syllableErrors['short_rising']]
        longErr = [str(i) for i in \
                result.syllableErrors['long_falling']]
        return '%s,%s' % (''.join(shortErr), ''.join(longErr))

    def getYesNoResult(self, theData):
        line = data.analysis.Line()
        line.content = data.text.Text(theData)
        result = self.analyzer.validate(line)
        if len(result.generalErrors) > 0:
            return '-'
        else:
            return 'ok'
    
if __name__ == "__main__":
    unittest.main()
