import data.analysis
import AnalyzerStrategy

MIN_LEN = 8
MAX_LEN = 10

class MatchAny:
    
    def accept(self, hyphenation, syllable, word, line, sylIndexInWord):
        return syllable != None
        
class MatchRise:
                                
    def accept(self, hyphenation, syllable, word, line, sylIndexInWord):
        mainStress = len(word.syllables) > 1 and sylIndexInWord == 0
        long = hyphenation.isLong(syllable)
        
        if mainStress:
            return long
        else:
            return True                

    def error(self, result, sylIndexInLine):
        result.syllableErrors['short_rising'].append(sylIndexInLine)

class MatchFall:
    
    def accept(self, hyphenation, syllable, word, line, sylIndexInWord):
        mainStress = len(word.syllables) > 1 and sylIndexInWord == 0
        long = hyphenation.isLong(syllable)
        
        if mainStress:
            return not long
        else:
            return True

    def error(self, result, sylIndexInLine):
        result.syllableErrors['long_falling'].append(sylIndexInLine)

class MatchLastSyllableNoLongVowel:

    def accept(self, hyphenation, syllable, word, line, sylIndexInWord):
        return not hyphenation.hasLongVowel(syllable)        

    def error(self, result, sylIndexInLine):
        result.generalErrors['long_vowel'] = 1

class MatchLastSyllableNoMonosyllable:
                                
    def accept(self, hyphenation, syllable, word, line, sylIndexInWord):
        return len(word.syllables) > 1

    def error(self, result, sylIndexInLine):
        result.generalErrors['monosyllable'] = 1

class HeavyBeginning:
        
    def accept(self, hyphenation, line):
        self.isError = False
                
        if line.content.numSyllables() != 9:
            return True
        
        beginning = [line.content.getSyllable(0),
                     line.content.getSyllable(1),
                     line.content.getSyllable(2)]
        
        self.isError = True
        for syll in beginning:
            if not hyphenation.isLong(syll):
                return True
            if not hyphenation.hasLongVowel(syll):
                self.isError = False
                
        return False

    def error(self, result):
        if self.isError:
            result.generalErrors['heavy_beginning_e'] = 1
        else:
            result.warnings['heavy_beginning_w'] = 1

class Caesura:
        
    def accept(self, hyphenation, line):
        if len(line.content.words) < 2:
            return True
        
        last = line.content.words[-1]
        secondToLast = line.content.words[-2]
        if len(last.syllables) == 2 and len(secondToLast.syllables) == 4:
            return False
        
        return True
        
    def error(self, result):
        result.generalErrors['caesura'] = 1        

class Analyzer:

    def __init__(self, hyphenation):
        self.strategy = AnalyzerStrategy.AnalyzerStrategy(self)
        self.hyph = hyphenation
        
        self.match = 10 * [[]]
        self.match[0] = [MatchAny()]
        self.match[1] = [MatchAny()]
        self.match[2] = [MatchAny()]
        self.match[3] = [MatchAny()]
        self.match[4] = [MatchRise()]
        self.match[5] = [MatchFall()]
        self.match[6] = [MatchRise()]
        self.match[7] = [MatchFall()]
        self.match[8] = [MatchRise()]
        self.match[9] = [MatchLastSyllableNoLongVowel(),
                         MatchLastSyllableNoMonosyllable()]
        
        self.otherConstraints = [HeavyBeginning(), Caesura()]

    def analyzeLine(self, line, syllabifications, stats, observers):
        self.strategy.analyzeLine(line, syllabifications, stats, 
                                  observers)

    def validate(self, line):
        result = data.analysis.AnalysisResult()
        self.storeType(line, result)
        self.storeAlliteration(line, result)   
        self.storeBrokenness(result)
        self.storeOtherComments(line, result)
        
        if line.content.numSyllables() < MIN_LEN:
            result.generalErrors['too_short'] = 1
            return result
        
        if line.content.numSyllables() > MAX_LEN:
            result.generalErrors['too_long'] = 1
            return result
            
        offset = MAX_LEN - line.content.numSyllables() - 1
        
        sylIndexInLine = 1 # 1-based
        for word in line.content.words:
            sylIndexInWord = 0
            for syllable in word.syllables:
                for constraint in self.match[sylIndexInLine + offset]:
                    if not constraint.accept(self.hyph, syllable, word, line,
                                             sylIndexInWord):
                        constraint.error(result, sylIndexInLine)
                sylIndexInLine += 1
                sylIndexInWord += 1
        
        for constraint in self.otherConstraints:
            if not constraint.accept(self.hyph, line):
                constraint.error(result)
                
        return result

    def storeType(self, line, result):
        mainType = ''
        detailedType = ''
        syllableShapes = ''
        for cpd in line.content.compoundWords:
            totalNumSyllables = 0
            for word in cpd.words:
                totalNumSyllables += len(word.syllables)
                for i, syllable in enumerate(word.syllables):
                    syllableShapes += self.hyph.syllableShape(syllable)
                    if self.hyph.isLong(syllable):
                        if i == 0:
                            detailedType += 'P'
                        else:
                            detailedType +=  'p'
                    else:
                        if i == 0:
                            detailedType += 'L'
                        else:
                            detailedType +=  'l'
            mainType += str(totalNumSyllables)
        result.mainType = mainType
        result.detailedType = detailedType
        result.syllableShapes = syllableShapes

    def storeAlliteration(self, line, result):
        consonantStrong = False
        vowelStrong = False
        consonantWeak = False
        vowelWeak = False
        
        for w1 in line.content.words:
            for w2 in line.content.words:
                if w1 == w2:
                    continue
                    
                syll1 = self.hyph.customLower(w1.syllables[0].phones)
                onset1 = self.hyph.getOnset(syll1)
                nucleus1 = self.hyph.getNucleus(syll1)
                syll2 = self.hyph.customLower(w2.syllables[0].phones)
                onset2 = self.hyph.getOnset(syll2)
                nucleus2 = self.hyph.getNucleus(syll2)
                
                if len(nucleus1) == 0 or len(nucleus2) == 0:
                    continue
                
                # V(C)
                if len(onset1) == 0:
                    if len(onset2) == 0:                        
                        if syll1[0] == syll2[0]:
                            vowelStrong = True
                        else:
                            vowelWeak = True
                # CV(C)
                else:
                    if len(onset2) > 0:
                        if onset1[0] == onset2[0]:
                            if onset1 == onset2 \
                            and nucleus1[0] == nucleus2[0]:
                                consonantStrong = True
                            else:
                                consonantWeak = True
                    
        if consonantStrong:        
            result.alliterationTypes += 'A'
        if vowelStrong:        
            result.alliterationTypes += 'B'
        if consonantWeak:
            result.alliterationTypes += 'C'
        if vowelWeak:        
            result.alliterationTypes += 'D'
            
    def storeBrokenness(self, result):
        pos = 9
        for wordLength in result.mainType[::-1]:
            pos -= int(wordLength)
            if int(wordLength) > 1 and pos in [2, 4, 6]:
                result.isBroken = True

    def storeOtherComments(self, line, result):
        if line.content.numSyllables() == 9:
            result.notes['nine'] = 1
        if line.content.numSyllables() == 10:
            result.notes['ten'] = 1