from Tkinter import *
import random
import data.text
import lang.Hyphenation
import engine.Preprocessor

SQUARE_W = 30
SQUARE_H = 20
ROWS = 17
COLS = 8
INIT_TICK = 900
SPEEDUP = .001
PT_BLOCK_DROPPED = 5
PT_LINE_CLEARED = 50

class Block:
    count = 0
    
    def __init__(self, text, canvas):
        self.text = text
        self.canvas = canvas
        self.tag = 'block' + str(Block.count)
        self.row = 0
        self.col = 0
        Block.count += 1
        
        x = self.col * SQUARE_W
        y = self.row * SQUARE_H
        for syllable in self.text.words[0].syllables:            
            self.canvas.create_rectangle(x, y, x + SQUARE_W, 
                    y + SQUARE_H, tags=self.tag)
            self.canvas.create_text(x + 1, y + 1, text=syllable.phones,
                    anchor=NW, tags=self.tag)
            x += SQUARE_W
        
    def move(self, rowDelta, colDelta):
        self.row += rowDelta
        self.col += colDelta
        self.canvas.move(self.tag, colDelta * SQUARE_W, 
                rowDelta * SQUARE_H)

    def drop(self, row):
        self.row = row
        
    def remove(self, canvas):
        canvas.delete(self.tag)
        
class Line:
    count = 0
    
    def __init__(self, row, canvas):
        self.row = row
        self.canvas = canvas
        self.syll = 8 * ['']
        self.isInitial = 8 * [False]
        self.tag = 'line' + str(Line.count)
        self.locked = False
        Line.count += 1
        self.__draw(canvas)
    
    def redraw(self, canvas):
        canvas.delete(self.tag)
        self.__draw(canvas)
        
    def collision(self, block, canvas, mainWin):
        invalidPos = self.locked
        for i in range(block.text.numSyllables()):
            if self.syll[block.col + i] != '':
                invalidPos = True
                
        if invalidPos:
            mainWin.newLine()
            mainWin.addPoints(-PT_BLOCK_DROPPED)
            self.locked = True
            mainWin.redrawLines()
            return
        
        mainWin.addPoints(PT_BLOCK_DROPPED)
                
        for i in range(block.text.numSyllables()):
            self.syll[block.col + i] = block.text.getSyllable(i).phones
            if i == 0:
                self.isInitial[block.col + i] = True
            else:
                self.isInitial[block.col + i] = False
        
        if len([x for x in self.syll if x == '']) == 0:
            if mainWin.valid(str(self)):
                mainWin.addPoints(PT_LINE_CLEARED)
                mainWin.removeLine(self) # causes redraw
                mainWin.newLine()
            else:
                self.locked = True
                mainWin.newLine()
                mainWin.redrawLines()
        else:
            self.redraw(canvas)
        
    def setRow(self, row, canvas):
        self.row = row
        self.redraw(canvas)
        
    def __draw(self, canvas):
        x = 0
        y = self.row * SQUARE_H
        if self.locked:
            bg = '#e4e4e4'
        else:
            bg = '#e4e4ff'
        for i, syllable in enumerate(self.syll):
            self.canvas.create_rectangle(x, y, x + SQUARE_W, 
                    y + SQUARE_H, tags=self.tag, fill=bg)
            self.canvas.create_text(x + 1, y + 1, text=syllable,
                    anchor=NW, tags=self.tag)
            if self.isInitial[i]:
                self.canvas.create_line(x + 1, y, x + 1, y + SQUARE_H,
                        tags=self.tag)
            x += SQUARE_W
            
    def __str__(self):
        strContent = ''
        for i, syllable in enumerate(self.syll):
            if self.isInitial[i]:
                strContent += ' '
            strContent += syllable
        return strContent
            
class MainWindow(Frame):              
    def __init__(self, master=None):
        Frame.__init__(self, master)
        self.currentBlock = None
        self.lines = [] # first item = bottom row
        self.running = False
        self.points = 0
        self.tickInterval = INIT_TICK
        self.hyph = lang.Hyphenation.Hyphenation()
        self.pre = engine.Preprocessor.Preprocessor()
        
        self.grid()                    
        self.createWidgets()
        self.bindKeys()

    def createWidgets(self):
        self.field = Canvas(self, width=COLS * SQUARE_W, 
                height=ROWS * SQUARE_H, borderwidth=0,
                background='#e4e4ff')
        self.field.grid(row=1, column=0)
        self.startButton = Button(self, text='start', command=self.start)
        self.startButton.grid(row=1, column=1)
        self.pointsLabel = Label(self, text='0')
        self.pointsLabel.grid(row=0, column=0) 
    
    def bindKeys(self):
        self.bind_all('<Key-Left>', self.__left)
        self.bind_all('<Key-Right>', self.__right)
        self.bind_all('<Key-Up>', self.__up)
        self.bind_all('<Key-Down>', self.__down)
        
    def start(self):
        if self.running:
            self.running = False
        else:
            for id in self.field.find_all():
                self.field.delete(id)
            self.lines = []
            self.currentBlock = Block(self.nextWord(), self.field)
            for i in range(4):
                self.newLine()
            self.points = 0    
            self.tickInterval = INIT_TICK
            self.after(int(self.tickInterval), self.tick)
            self.running = True
        
    def tick(self):
        if not self.running:
            return

        self.currentBlock.move(1, 0)
        self.__checkCollision()
        self.__checkGameOver()
        self.tickInterval -= SPEEDUP * self.tickInterval
        self.after(int(self.tickInterval), self.tick)
        
    def nextWord(self):
        f = open('../data/kalevala_vain_sakeet.txt')
        #lineToRead = random.randrange(22795)
        lineToRead = random.randrange(2)
        lineIndex = 0
        for line in f:
            if lineIndex == lineToRead:
                hyphLine = self.hyph.hyphenate(
                        self.pre.process(line.lower()))
                wordToRead = random.choice(hyphLine.split(' '))
                return phono.text.Text(wordToRead)
            lineIndex += 1
        return ''

    # called from Line
    
    def newLine(self):
        self.lines.append(Line(ROWS - 1 - len(self.lines), self.field))
    
    def removeLine(self, lineToRemove):
        print 'removing ' + str(lineToRemove)
        self.lines.remove(lineToRemove)
        for i, line in enumerate(self.lines):
            line.setRow(ROWS - 1 - i, self.field)
    
    def addPoints(self, points):
        self.points += points
        self.pointsLabel.config(text=str(self.points))
    
    def redrawLines(self):
        for line in self.lines:
            line.redraw(self.field)
            
    def valid(self, text):
        print 'validating ' + text
        return True
    
    #
    
    def __checkCollision(self):
        if self.currentBlock.row >= self.lines[-1].row:
            self.lines[-1].collision(self.currentBlock, self.field, self)
            self.currentBlock.remove(self.field)
            self.currentBlock = Block(self.nextWord(), self.field)
            
    def __checkGameOver(self):
        if len(self.lines) >= ROWS - 1:
            self.running = False
            
    def __left(self, event):
        if self.currentBlock.col > 0:
            self.currentBlock.move(0, -1)
        
    def __right(self, event):
        if self.currentBlock.col \
        + self.currentBlock.text.numSyllables() < COLS:
            self.currentBlock.move(0, 1)
    
    def __up(self, event):
        self.lines = self.lines[-1:] + self.lines[:-1]
        for i, line in enumerate(self.lines):
            line.setRow(ROWS - 1 - i, self.field)
            
    def __down(self, event):
        if self.currentBlock.row < self.lines[0].row - 1:
            self.currentBlock.drop(self.lines[0].row - 1)